### A Pluto.jl notebook ###
# v0.20.4

using Markdown
using InteractiveUtils

# ╔═╡ 0cd4827b-b479-47ae-8ad3-f825a1ac9948
# ╠═╡ show_logs = false
import Pkg; Pkg.activate(".")

# ╔═╡ 6567291b-91ba-45b4-b625-34e2a2b4caf6
begin
	using AbstractFFTs
	using CairoMakie
	using CommonMark
	using CSV
	using DataFrames
	using DensityInterface
	using DSP
	using Format
	using HypothesisTests
	using Latexify
	using Optim
	using PairPlots
	import Plots
	using PlutoUI
	using PValue
	using Statistics
	using StateSpaceModels
	using StatsBase
	using StatsPlots
	using Turing
end

# ╔═╡ 2e053d2a-7cc0-4df7-bda1-f43b7d397518
md"""
**What is this?**


*This jupyter notebook is part of a collection of notebooks on various topics discussed during the Time Domain Astrophysics course delivered by Stefano Covino at the [Università dell'Insubria](https://www.uninsubria.eu/) in Como (Italy). Please direct questions and suggestions to [stefano.covino@inaf.it](mailto:stefano.covino@inaf.it).*
"""

# ╔═╡ b1ea7402-3509-46f1-aefb-c2d51215062c
md"""
**This is a `Julia` notebook**
"""

# ╔═╡ 2ceeb12e-e068-4401-a921-bb45a2396424
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/TimeDomainBanner.jpg"))
"""

# ╔═╡ cb93de12-47c8-40e8-a056-4a4f08d42d66
# ╠═╡ show_logs = false
md"""
# Science Case: AGN and Blazars

***

- Why are they special?
- What is the power source?
- What is the sources of continuum emission?
- What is producing emission and absorption features?
- How do jets form?
- AGN classification schema and unification.
- The co-evolution of black holes and galaxies.



- More specifically:
    - Very large luminosities are possible, up to 10,000 times a typical galaxy.
    - The emission spans a huge range of photon energy, i.e. from radio to gamma-rays.
    - The source of energy generation is very compact, typically lower than the size of the solar system.
    - In some cases, there is significant energy transported in relativistic jets.



- This AGN, [3C273](https://simbad.u-strasbg.fr/simbad/sim-id?Ident=3C+273), is several hundred times brighter than its host galaxy, just in visible light alone:

$(LocalResource("Pics/3c273.jpg"))

"""

# ╔═╡ 3f25cd6c-9b2d-4459-9651-7c1df7be5f2c
# ╠═╡ show_logs = false
md"""
### AGN: Observational data
***

- Highly luminous: $L_{\rm bol} \sim 10^{42}–10^{48}$ erg s$^{-1}$ ($10^9-10^{15} L_\odot$).

- Compact: size << 1pc.

- Broad-band continuum emission:  
    - dL/dlog f = const.  From IR to X-rays and γ-rays

- Variable: on different times scales. Rapid variability in the X-ray indicate that emission comes from the innermost region.

- Strong Radio emitters: in some sources extended, jets are present.

- Polarized emission.



- [Markarian 421](https://simbad.u-strasbg.fr/simbad/sim-id?Ident=Mrk++421) shows comparable power emitted across abouit seven orders of magnitude in photon energy:

$(LocalResource("Pics/mrk421.png"))

"""

# ╔═╡ 345f1ef8-e930-453a-aac0-c44cffb76d44
# ╠═╡ show_logs = false
md"""
- [Markarian 421](https://simbad.u-strasbg.fr/simbad/sim-id?Ident=Mrk++421) is also bright at TeV energies, nowdays accessible, e.g., by [Cherenkov telescopes](https://en.wikipedia.org/wiki/IACT):

$(LocalResource("Pics/mrk421tev.png"))
"""

# ╔═╡ a8f9e126-1730-4f30-8d70-9b454f56ff39
# ╠═╡ show_logs = false
md"""
- Light travel time argument: a source that varies significantly in time t must have size  $R < ct$, where $c$ is the speed of light.

$(LocalResource("Pics/compactsource.png"))
"""

# ╔═╡ 7e8db3bd-fac0-44a3-b4f3-5246cc2eda89
md"""
#### Exercize: compute the upper limit to the source size for [Mrk 421](https://simbad.u-strasbg.fr/simbad/sim-id?Ident=Mrk++421):
***

- At $\sim 100$ keV the shortest observed variability is $\sim 2.9$ hours.

- The size of the emitting region turns out to be:

```math
l \le \frac{c \Delta t \delta}{1+z}
```

- where $c$ is the speed of light, $\Delta t$ the variability time, $\delta$ the Doppler factor and $z$ the source redshift. 

- The Doppler factor is $\delta = \frac{1}{\Gamma (1-\beta \cos \theta)} = \frac{\sqrt{1-\beta2}}{1-\beta \cos \theta}$, where $\Gamma$ is the Loremtz factor, $\beta$ the source speed in units of the speed of light, and $\theta$ the angle of motion wrt the line of sight.
    - For this source $z = 0.030$, $\theta \approx 0^\circ$ and $\beta > 0.98$ so that $\delta \sim 10$ and, finally, $l \le 3 \times 10^{15}$ cm = 203 AU.
"""

# ╔═╡ 3586aabc-e618-4247-827b-777224115314
# ╠═╡ show_logs = false
cm"""
## The AGN unified model
***

- Many of the various phenomenologies shown by AGN can be explained by a unique simple scenario and dfferent viewing angles:

$(LocalResource("Pics/unifiedmodel.png"))



- The power source is accretion onto a supermassive black-hole. 
    - Efficient, compact, and capable of producing high-energy emission and jets.



- It is occasionally possible to measure directly the mass of the central black-hole, e.g., by Newtonian dynamics (``M = v^2 R/G``):
    - Water masers have been mapped in [NGC 4258](https://simbad.u-strasbg.fr/simbad/sim-id?Ident=NGC+4258):  ``M \sim 40 \times 10^6 M_\odot``. 
    - Orbits of stars in the [Galactic Center](https://en.wikipedia.org/wiki/Galactic_Center): ``M \sim 3 \times 10^6 M_\odot``.

| $(LocalResource("Pics/rotcurve.jpg")) | $(LocalResource("Pics/galcenter.jpg")) |
| ------------------------------------ | -------------------------------------- |



- And now we also have direct imaging of the [M87](https://simbad.u-strasbg.fr/simbad/sim-id?Ident=M87) central black holes:

$(LocalResource("Pics/m87.png"))
"""

# ╔═╡ edfcfc7f-e288-4c9f-b11b-b4cda0553a78
# ╠═╡ show_logs = false
md"""
### A few more thoughts
***

- AGN are important for several more reasons:
    - They have produced ~10% of all the luminous energy since the Big Bang.
    - They are unique laboratories for studying physics under extreme conditions.
    - They played a major role in the evolution of the baryonic component of the universe (galaxies and the inter-galactic medium).



- Among the variuous AGN phenomenologies, jets play a fundamental role:

    -  If one of the two opposite jets is pointing to the observer (i.e. Earth) we call these sources “blazars”.
    - There are, as usual, subtypes, but in general the jet emission outshines the whole galaxy by a large factor. 
    - Plenty of relativistic phenomena to take into account. Blazars dominate the high-energy sky.



- These sorces are often monitored since decades with with multi-band light curves.
    - This is, for instance, the dataset obtained by the [INAF](http://www.inaf.it/en?set_language=en) [REM telescope](https://en.wikipedia.org/wiki/Rapid_Eye_Mount_telescope) for [PKS2155-304](https://simbad.u-strasbg.fr/simbad/sim-basic?Ident=PKS+2155-304&submit=SIMBAD+search):
   
$(LocalResource("Pics/pks2155-304-rem.jpg"))
"""

# ╔═╡ 1c3ac56a-d566-44e6-bf16-be9934724a77
# ╠═╡ show_logs = false
md"""
### AGN/Blazar with binary central black-holes?
***

- There is strong interest in (possible) blazar periodicities.
- It is a highly debated topic, a general agreement in the community is still missing for a series of reasons we are now able to realize:
    - Often, we refer to year-long (quasi-)periodicities, with light-curve at most a couple of decades long. Usual statistical tests are almost ineffecitve. 
    - Time-series are affected by red-noise, making evaluation of the significance of identified features difficult.
    - For data from ground-based facilities sampling is irregular with long (seasonal) gaps.
    
$(LocalResource("Pics/binary.jpg"))
"""

# ╔═╡ c83c4c6c-b3c3-48a5-973f-ae68427c0972
md"""
#### Exercise: [PG1553+113](https://simbad.u-strasbg.fr/simbad/sim-id?Ident=PG+1553%2B113) claimed periodicity
***

- We analyse data obtained by the [Fermi satellite](https://fermi.gsfc.nasa.gov/) with the [LAT](https://fermi.gsfc.nasa.gov/science/instruments/lat.html) instrument. Data can be downloaded [here](https://fermi.gsfc.nasa.gov/ssc/data/access/).

- Let's see the light-curve:
"""

# ╔═╡ f502d84a-7f71-4c5d-b1c1-fb975ee61b8c
begin
	dt = DataFrame(CSV.File("4FGLJ1555 7p1111.txt",select=[1,2,3],comment="#"))
	rename!(dt,["MJD","Rate","eRate"])
	dt[!,:Rate] =  dt[!,:Rate] * 1e7
	dt[!,:eRate] =  dt[!,:eRate] * 1e7
end;

# ╔═╡ 2cdbd150-ccd5-4f2a-9e49-5321070a2208
begin
	fg1 = Figure()
	
	axfg1 =  Axis(fg1[1,1],
	    xlabel="Time (MJD)",
	    ylabel="Flux (arbitrary units)",
	    )
	
	CairoMakie.scatter!(dt[!,:MJD],dt[!,:Rate],label="Fermi data for PG1553+113")
	CairoMakie.errorbars!(dt[!,:MJD],dt[!,:Rate],dt[!,:eRate])
	
	axislegend()
	
	fg1
end

# ╔═╡ 21f1efc6-2e79-4c06-862d-55b5d20966d5
md"""
- The light-curve shows a possible cyclic behavior. This is indeed one of the sources of this category with the strongest claims for a possible periodicity.

- We first try to model this time-series with an ARIMA model. Data are (approximately) evenly sampled with a point in 30 days.
"""

# ╔═╡ 94e01a40-8bd6-469a-b348-7aba8a6280d4
begin
	meandelta = mean(diff(dt[!,:MJD]))
	meandeltabin = std(diff(dt[!,:MJD]))
	#printfmtln("Mean sample rate: {:.2f} days, with standard deviation: {:.2f} days",meandelta, meandeltabin)
end;

# ╔═╡ 5a1bc2de-a335-4024-a249-fec980636cdc
Markdown.parse("""
Mean sample rate: $(latexify(meandelta,fmt="%.2f")) days, with standard deviation: $(latexify(meandeltabin,fmt="%.2f")) days.
""")

# ╔═╡ 8af78ef0-28f1-4222-bd62-1df8dfe4b931
md"""
- Let's first check data stationarity.
"""

# ╔═╡ dc922b83-b899-41df-ba15-32c85ad65b2b
ADFTest(dt[!,"Rate"],:constant,1)

# ╔═╡ 70b1d915-2425-4741-9dd9-b7d60496b4ba
md"""
- The light-curve can be considered, at least during this exercise, as stationary and therefore we go on modeling it with an ARMA model.

- First, let's look at the ACF and PACF diagnostic plots.
"""

# ╔═╡ 31f409e8-9c25-4b30-b768-89a4aab4546e
begin
	tracf = GetACF(dt[!,:Rate],50)
	trpacf = GetPACF(dt[!,:Rate],50)
	
	fg2 = Figure()
	ax1fg2 = Axis(fg2[1, 1],
	    title = "ACF")
	
	stem!(tracf["ACF"])
	hlines!([tracf["errACFmin"],tracf["errACFmax"]],linestyle=:dash)
	
	ax2fg2 = Axis(fg2[2, 1],
	    title = "PACF")
	
	stem!(trpacf["PACF"])
	hlines!([trpacf["errPACFmin"],trpacf["errPACFmax"]],linestyle=:dash)
	
	fg2
end

# ╔═╡ d9f29b49-fcdf-4762-b1ca-795d2ad62d67
md"""
- The ACF is zero after the first 7 lags, but it also shows the typical cyclic behavior of periodic phenomena, with a period of about 25 lags $\sim 2$ years that is close to the proposed (quasi-)periodicity of 2.2 years ([Achermann et al. 2015](https://ui.adsabs.harvard.edu/abs/2015ApJ...813L..41A/abstract)).

- The PACF is zero after 3 lags. This might suggest 3 as AR order and 7 as MA order, altgough we prefer to look for the best solution by a grid search.
"""

# ╔═╡ 26f04717-948d-4e3d-be26-e6e8c6070905
begin
	bics = Dict()
	minbc = 1e6
	for p in 0:5
	    for q in 0:5
	        model_ARIMA = StateSpaceModels.SARIMA(dt[!,:Rate]; order = (p, 0, q), suppress_warns=true)
	        try
	            StateSpaceModels.fit!(model_ARIMA, save_hyperparameter_distribution=false, optimizer = Optimizer(StateSpaceModels.Optim.NelderMead()))
	            println("p: ", p, " q: ", q, " BIC: ", model_ARIMA.results.bic)
	            if model_ARIMA.results.bic < minbc
	                minbc = model_ARIMA.results.bic
	                bics["BIC"] = minbc
	                bics["p"] = p
	                bics["q"] = q
	            end
	        catch DomainError
	            print()
	        end
	    end
	end
	println(bics)
end

# ╔═╡ ef6c457b-9f04-45b7-bf31-281a39111fb6
md"""
- Results are indeed not so close to those inferred from the ACF/PACF analysis. Let's see some diagnostic plots.
"""

# ╔═╡ b8849f80-f454-464e-881b-4b5a9f2dcbdd
begin
	model_ARIMA = StateSpaceModels.SARIMA(dt[!,:Rate]; order = (bics["p"], 0, bics["q"]), suppress_warns=true)
	StateSpaceModels.fit!(model_ARIMA, save_hyperparameter_distribution=false, optimizer = Optimizer(StateSpaceModels.Optim.NelderMead()))
end;

# ╔═╡ 3fdda80e-3360-481c-af16-0bcdd5d3e470
begin
	kf = kalman_filter(model_ARIMA)
	
	plotdiagnostics(kf)
end

# ╔═╡ 29e38a84-e43f-434a-83cf-3003a33b1079
md"""
- And we immediately see that we have discrepancies in particular for the high flixes. This might be due to the hint of periodicity that cannot be described by a pure stochastic model as the ARMA process we are testing.
"""

# ╔═╡ 7dbbeaad-3631-4e42-bce0-38ff0d1f2b0f
begin
	fg3 = Figure()
	
	ax1fg3 = Axis(fg3[1, 1],
	    title="ARMA fit",
	    )
	
	lines!(model_ARIMA.system.y,color=:blue,label="data")
	lines!(model_ARIMA.system.y .+ StateSpaceModels.get_innovations(kf)[:,1],color=:red,alpha=0.5)
	
	axislegend()
	
	fg3
end

# ╔═╡ 604dab44-6d5f-49e3-a2c1-669a947ac5ea
md"""
- The ARMA model is indeed rather effective in gettig the variability of the data, but it fails to reproduce the extent of the variability, in particular at the highest or lowest fluxes.

- We then try to apply a more sophisticated analysis, based on a procedure described by [Vaughan (2010)](https://ui.adsabs.harvard.edu/abs/2010MNRAS.402..307V/abstract).
"""

# ╔═╡ a862c4a2-0a3c-432d-86ab-49c9857554b3
md"""
- Let's compuyte the Nyquist frequency and a lower-limit for the freqencies accessible with our dataset:
"""

# ╔═╡ 15fd10c7-c163-4c4c-a1a7-993ec5185843
begin
	lendata = maximum(dt.MJD)-minimum(dt.MJD)
	nqy = 1 / (2*meandelta)
	#printfmtln("Nyquist frequency: {:.4f}, lowest frequency: {:.4f} 1/day", nqy, 1/lendata)
end;

# ╔═╡ 83fa92c8-643c-408c-9d05-bc8d744e1e1a
Markdown.parse("""
##### Nyquist frequency: $(latexify(nqy,fmt="%.4f")), lowest frequency:  $(latexify(1/lendata,fmt="%.4f"))
""")

# ╔═╡ 520a01ff-dfca-405a-8cff-4124f0bc0792
md"""
- And let's compute the periodogram indicating the expected Poissonian noise.
"""

# ╔═╡ be5706a4-0325-45ed-b1b1-bab78d6485e7
begin
	psd = periodogram(dt.Rate; fs=1/meandelta)
	
	freq = psd.freq[psd.freq.>0]
	power = psd.power[psd.freq.>0]/sum(dt.eRate.^2);
end;

# ╔═╡ 8678584b-0bee-4ff0-a466-9f073847d2a1
begin
	fg4 = Figure()
	
	axfg4 = Axis(fg4[1,1],
	    title="Fourier periodogram",
	    ylabel="Power",
	    xlabel=L"Frequency (day$^{-1}$)",
	    xscale = log10,
	    yscale = log10
	    )
	
	lines!(freq,power)
	hlines!(2.,linestyle=:dash)
	
	
	fg4
end

# ╔═╡ a8aa2b22-1529-48da-beb0-b51cce0ae2aa
md"""
- We plan to model the noise of the time-series as PL plus a constant, i.e. the Poissonian noise level. The PL+constant model is the null hypotesis of a time-series generated by noise only without any periodicity.
"""

# ╔═╡ 2c1416c7-a866-4742-a424-233eca091657
function model(theta, x)
    lnn,a,g = theta
    return exp.(lnn) .* x.^a .+ abs.(g)
end

# ╔═╡ 974e371c-c0b2-40db-b7a9-ad19612ab76c
md"""
- We are going to carry out a Bayesian analysis and we need to define a proper likelihood. In this case, due to the statistics affecting a periodogram, we work with the *Whittle likelihood*.
"""

# ╔═╡ 2d72fd9b-25e4-46d0-ad3b-3f35ae4e2c09
function lnlike(theta, x, y)
    lnn,a,g = theta
    return -2.0 .* sum(y ./ model(theta,x) .+ log.(model(theta,x)))
end

# ╔═╡ fc38a1b7-51e4-4832-bfdb-e6d83af0ed27
@model function permodel(x,y)
    #
    lnn ~ Uniform(-15,0)
    a ~ Truncated(Normal(0.,10.),-10,0)
    g ~ Normal(2,0.4)
    #
    Turing.@addlogprob! lnlike((lnn,a,g),x,y)
    #
    for i in 1:length(y)
        y[i] ~ Chisq(2) * model((lnn,a,g),x[i]) / 2
    end
    #
end

# ╔═╡ 1d86c6a4-36ab-455a-a6b8-618fe8e36665
md"""
- Now, let's prepare and run the MCMC chain.
"""

# ╔═╡ 47a41ecc-faeb-45f7-a856-03c1d5bb917b
begin
	iterations = length(freq)*250
	burnins = 1000
	chains = 4
	
	chain = mapreduce(c -> sample(permodel(freq,power), NUTS(burnins,0.65), iterations), chainscat, 1:chains)
end

# ╔═╡ 2abc083b-f39f-4610-9c40-cfbfb73982a3
StatsPlots.plot(chain)

# ╔═╡ eb11b179-5c4c-4b29-b90d-6fd1cf1c54b4
md"""
- And let's check the chain result by a classic *corner plot*!
"""

# ╔═╡ c9b3afdd-b00f-4735-ab7e-862e9fcac505
begin
	dfc = DataFrame(chain)     
	
	dfc[!,:Lnn] = log10.(exp.(dfc[!,:lnn]))
	          
	dfcs = select(dfc, [:Lnn,:a, :g])
	
	fig = pairplot(dfcs, labels = Dict(
	        :Lnn => "Log N",
	        :a => "α",  
	        :g => "g"
	    ))
end

# ╔═╡ 39235756-f5b9-4e0d-a47b-64235767ca4c
md"""
- The posteriors are well defined and single-peaked.

- Since the statistics is $\chi^2$ with 2 degrees of freedom we can check whether the model is properly fitting the data, as it seems to be the case.
"""

# ╔═╡ 593ff4f2-6fb9-4031-8a5b-d7c83838ecf7
begin
	theta = [mean(chain["lnn"]),mean(chain["a"]),mean(chain["g"])]
	
	ExactOneSampleKSTest(2*power./model(theta,freq), Chisq(2))
end

# ╔═╡ 25258185-5a00-4faa-ad78-fdb2e770909a
md"""
- Now, let's compute, basing upon the MSMC chains, the percentile for any chosen level for each frequency bin. Details of the procedure we follow are discussed in [Vaughan (2010)](https://ui.adsabs.harvard.edu/abs/2010MNRAS.402..307V/abstract).
"""

# ╔═╡ 4bc8b53c-b9f6-47b8-ab9e-9cc2871488e1
begin
	# Compute the best fit model
	Sbest = model(theta,freq)
	
	# Derive a set of models based on the posterrio distribution
	S = [model((dfc[i,:lnn],dfc[i,:a],dfc[i,:g]),freq) for i in 1:nrow(dfc)]
	
	# Multiply a chi2 distribution to the models and derive the simulated periodograms
	I = S .* rand(Chisq(2),length(S))/2.
	
	# Compute the observed
	Robs = 2*power./Sbest
end;

# ╔═╡ a3d22608-bb4f-49db-bc4d-9086a2cf192e
begin
	# Compute the simulated Rs
	Is = reshape(I,(Int(length(I)/length(Sbest)),length(Sbest)))
	Rsim = [2*Is[i,:][1] ./ Sbest for i in 1:size(Is)[1]]
end;

# ╔═╡ f7e7a22c-6ef0-4910-9deb-4901aa917c99
begin
	# And derive any chosen percentile on the simulated distribution
	R95 = percentile([maximum(Rsim[i]) for i in 1:length(Rsim)],95.0)
	R997 = percentile([maximum(Rsim[i]) for i in 1:length(Rsim)],99.7)
	
	# And convert back to periodogram values.
	pp95 = R95.*Sbest/2.
	pp997 = R997.*Sbest/2.
end;

# ╔═╡ 3185c0e0-edcb-46b0-9479-de6d84ce08c0
md"""
- And finally let's plot our periodogram together with the predictions for pure noise datasets.
"""

# ╔═╡ e8829c5d-2cf7-4303-9ad4-1c457ca734bd
begin
	fg5 = Figure()
	
	axfg5 = Axis(fg5[1,1],
	    title = "PKS2155-304",
	    xlabel = L"Frequency (day$^{-1}$)",
	    ylabel = "Power",
	    xscale=log10,
	    yscale=log10,
	    )
	    
	
	lines!(freq,power,label="data",color=:blue)
	lines!(freq,Sbest,label="fit",color=:red)
	lines!(freq,pp95,label="95%",color=:magenta)
	lines!(freq,pp997,label="99.7%",color=:orange)
	
	axislegend()
	
	#plt.ylim((0.01,3e3));
	
	fg5
end

# ╔═╡ 4f538581-25a3-4aa2-8029-ecebf4234b9e
md"""
- The peak in the periodogram, although of interest, does not reach the 3$\sigma$ limit threshold, and we cannot rule out the possibility it is a noise artifact. 
"""

# ╔═╡ d66e12db-ce72-41a6-8207-2533bee71604
md"""
## Reference & Material

Material and papers related to the topics discussed in this lecture.

- [Covino et al. (2019) - "Gamma-ray quasi-periodicities of blazars. A cautious approach"](https://ui.adsabs.harvard.edu/abs/2020ApJ...895..122C/abstract)
"""

# ╔═╡ 14f46e3f-17ba-48a2-8ed1-8ebfe9b3991c
md"""
## Further Material

Papers for examining more closely some of the discussed topics.

- [Ackermann et al. (2015) - "Multiwavelength Evidence for Quasi-periodic Modulation in the Gamma-Ray Blazar PG 1553+113"](https://ui.adsabs.harvard.edu/abs/2015ApJ...813L..41A/abstract)
- [Vaughan (2010) - "A Bayesian test for periodic signals in red noise"](https://ui.adsabs.harvard.edu/abs/2010MNRAS.402..307V/abstract)
"""

# ╔═╡ fb3021c6-b871-47e2-acf2-4b4f31ae18a2
md"""
### Credits
***

This notebook contains no exertnal material. 
"""

# ╔═╡ 1a698a3e-bb7e-46c9-b045-fc12d21f41ab
cm"""
## Course Flow

<table>
  <tr>
    <td>Previous lecture</td>
    <td>Next lecture</td>
  </tr>
  <tr>
    <td><a href="./open?path=Lectures/Lecture - Time Domain Analysis/Lecture-Time-Domain.jl">Lecture about time domain analysis</a></td>
    <td><a href="./open?path=Lecture - Wavelet Analysis/Lecture-Wavelet-Analysis.jl">Lecture about wavelet analysis</a></td>
  </tr>
 </table>


"""

# ╔═╡ 5f3ce6b3-4df7-4928-b01e-9a4e4849f673
md"""
**Copyright**

This notebook is provided as [Open Educational Resource](https://en.wikipedia.org/wiki/Open_educational_resources). Feel free to use the notebook for your own purposes. The text is licensed under [Creative Commons Attribution 4.0](https://creativecommons.org/licenses/by/4.0/), the code of the examples, unless obtained from other properly quoted sources, under the [MIT license](https://opensource.org/licenses/MIT). Please attribute the work as follows: *Stefano Covino, Time Domain Astrophysics - Lecture notes featuring computational examples, 2024*.
"""

# ╔═╡ Cell order:
# ╟─2e053d2a-7cc0-4df7-bda1-f43b7d397518
# ╟─b1ea7402-3509-46f1-aefb-c2d51215062c
# ╠═0cd4827b-b479-47ae-8ad3-f825a1ac9948
# ╠═6567291b-91ba-45b4-b625-34e2a2b4caf6
# ╟─2ceeb12e-e068-4401-a921-bb45a2396424
# ╟─cb93de12-47c8-40e8-a056-4a4f08d42d66
# ╟─3f25cd6c-9b2d-4459-9651-7c1df7be5f2c
# ╟─345f1ef8-e930-453a-aac0-c44cffb76d44
# ╟─a8f9e126-1730-4f30-8d70-9b454f56ff39
# ╟─7e8db3bd-fac0-44a3-b4f3-5246cc2eda89
# ╟─3586aabc-e618-4247-827b-777224115314
# ╟─edfcfc7f-e288-4c9f-b11b-b4cda0553a78
# ╟─1c3ac56a-d566-44e6-bf16-be9934724a77
# ╟─c83c4c6c-b3c3-48a5-973f-ae68427c0972
# ╠═f502d84a-7f71-4c5d-b1c1-fb975ee61b8c
# ╠═2cdbd150-ccd5-4f2a-9e49-5321070a2208
# ╟─21f1efc6-2e79-4c06-862d-55b5d20966d5
# ╠═94e01a40-8bd6-469a-b348-7aba8a6280d4
# ╟─5a1bc2de-a335-4024-a249-fec980636cdc
# ╟─8af78ef0-28f1-4222-bd62-1df8dfe4b931
# ╠═dc922b83-b899-41df-ba15-32c85ad65b2b
# ╟─70b1d915-2425-4741-9dd9-b7d60496b4ba
# ╟─31f409e8-9c25-4b30-b768-89a4aab4546e
# ╟─d9f29b49-fcdf-4762-b1ca-795d2ad62d67
# ╠═26f04717-948d-4e3d-be26-e6e8c6070905
# ╟─ef6c457b-9f04-45b7-bf31-281a39111fb6
# ╠═b8849f80-f454-464e-881b-4b5a9f2dcbdd
# ╠═3fdda80e-3360-481c-af16-0bcdd5d3e470
# ╟─29e38a84-e43f-434a-83cf-3003a33b1079
# ╠═7dbbeaad-3631-4e42-bce0-38ff0d1f2b0f
# ╟─604dab44-6d5f-49e3-a2c1-669a947ac5ea
# ╟─a862c4a2-0a3c-432d-86ab-49c9857554b3
# ╠═15fd10c7-c163-4c4c-a1a7-993ec5185843
# ╟─83fa92c8-643c-408c-9d05-bc8d744e1e1a
# ╟─520a01ff-dfca-405a-8cff-4124f0bc0792
# ╠═be5706a4-0325-45ed-b1b1-bab78d6485e7
# ╠═8678584b-0bee-4ff0-a466-9f073847d2a1
# ╟─a8aa2b22-1529-48da-beb0-b51cce0ae2aa
# ╠═2c1416c7-a866-4742-a424-233eca091657
# ╟─974e371c-c0b2-40db-b7a9-ad19612ab76c
# ╠═2d72fd9b-25e4-46d0-ad3b-3f35ae4e2c09
# ╠═fc38a1b7-51e4-4832-bfdb-e6d83af0ed27
# ╟─1d86c6a4-36ab-455a-a6b8-618fe8e36665
# ╠═47a41ecc-faeb-45f7-a856-03c1d5bb917b
# ╠═2abc083b-f39f-4610-9c40-cfbfb73982a3
# ╟─eb11b179-5c4c-4b29-b90d-6fd1cf1c54b4
# ╠═c9b3afdd-b00f-4735-ab7e-862e9fcac505
# ╟─39235756-f5b9-4e0d-a47b-64235767ca4c
# ╠═593ff4f2-6fb9-4031-8a5b-d7c83838ecf7
# ╟─25258185-5a00-4faa-ad78-fdb2e770909a
# ╠═4bc8b53c-b9f6-47b8-ab9e-9cc2871488e1
# ╠═a3d22608-bb4f-49db-bc4d-9086a2cf192e
# ╠═f7e7a22c-6ef0-4910-9deb-4901aa917c99
# ╟─3185c0e0-edcb-46b0-9479-de6d84ce08c0
# ╠═e8829c5d-2cf7-4303-9ad4-1c457ca734bd
# ╟─4f538581-25a3-4aa2-8029-ecebf4234b9e
# ╟─d66e12db-ce72-41a6-8207-2533bee71604
# ╟─14f46e3f-17ba-48a2-8ed1-8ebfe9b3991c
# ╟─fb3021c6-b871-47e2-acf2-4b4f31ae18a2
# ╟─1a698a3e-bb7e-46c9-b045-fc12d21f41ab
# ╟─5f3ce6b3-4df7-4928-b01e-9a4e4849f673
